/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#ifndef _IRENDERER_H_
#define _IRENDERER_H_

#include <string>
#include "LayerType.h"
#include "Surface.h"
#include "Shader.h"
#include "OptimizationType.h"


class InputManager;


/**
 * Abstract Base of all CompositingControllers, ie Renderers.
 * \defgroup RendererAPI Layer Management Renderer API
 */
class IRenderer
{
public:
    /**
     * \brief      default destructor
     */
    virtual ~IRenderer()
    {
    }

    /**
     * \brief      Start the actual rendering process (render loop)
     * \ingroup    RendererAPI
     * \param[in]  width width of display handled by this renderer
     * \param[in]  height height of display handled by this renderer
     * \param[in]  displayName name of display handled by this renderer
     * \param[in]  maxIterationDurationInMS expected wakup time for health reporting in milliseconds (-1 to disable wakeup)
     * \return     TRUE: renderer was started successfully
     * \return     FALSE: renderer start failed
     */
    virtual bool start(int width, int height, const char* displayName, int maxIterationDurationInMS) = 0;

    /**
     * \brief      Stop rendering process (stop render loop)
     * \ingroup    RendererAPI
     */
    virtual void stop() = 0;

    /**
     * \brief      Switch debug mode of this component on or off
     * \ingroup    RendererAPI
     * \param[in]  onoff TRUE: Turn on debug mode, FALSE: Turn off debug mode
     */
    virtual void setdebug(bool onoff) = 0;

    /**
     * \brief      Store graphical content of screen to bitmap
     * \ingroup    RendererAPI
     * \param[in]  fileToSave path to bitmap file to store the graphical content
     * \param[in]  screen_id id of screen to dump
     */
    virtual void doScreenShot(std::string fileToSave, const unsigned int screen_id) = 0;

    /**
     * \brief      Store graphical content of layer to bitmap
     * \ingroup    RendererAPI
     * \param[in]  fileToSave path to bitmap file to store the graphical content
     * \param[in]  id id of layer
     */
    virtual void doScreenShotOfLayer(std::string fileToSave, const unsigned int id) = 0;

    /**
     * \brief      Store graphical content of surface to bitmap
     * \ingroup    RendererAPI
     * \param[in]  fileToSave path to bitmap file to store the graphical content
     * \param[in]  id id of surface
     * \param[in]  layer_id id of layer
     */
    virtual void doScreenShotOfSurface(std::string fileToSave, const unsigned int id, const unsigned int layer_id) = 0;

    /**
     * \brief      Get the capabilies of a layer type
     * \ingroup    RendererAPI
     * \param[in]  layertype type of layer
     * \return     bitset with flags set for capabilities
     */
    virtual unsigned int getLayerTypeCapabilities(LayerType layertype) = 0;

    /**
     * \brief      Get the number of supported hardware layers of the renderer for a screen
     * \ingroup    RendererAPI
     * \param[in]  screenID id of the screen
     * \return     Number of supported hardware layers for screen
     */
    virtual unsigned int getNumberOfHardwareLayers(unsigned int screenID) = 0;

    /**
     * \brief      Get the resolution of a screen handled by this renderer
     * \ingroup    RendererAPI
     * \param[in]  screenID id of the screen
     * \return     array with width and height of screen
     */
    virtual unsigned int* getScreenResolution(unsigned int screenID) = 0;

    /**
     * \brief      Get the list if available screen ids
     * \ingroup    RendererAPI
     * \param[out] length length of the returned array
     * \return     array containing all available screen ids
     */
    virtual unsigned int* getScreenIDs(unsigned int* length) = 0;

    /**
     * \brief      Create a shader object (that can be applied to surfaces)
     * \ingroup    RendererAPI
     * \param[in]  vertexName filename of vertex shader source code
     * \param[in]  fragmentName filename of fragment shader source code
     * \return     Pointer to created shader object
     */
    virtual Shader* createShader(const string* vertexName, const string* fragmentName) = 0;

    /**
     * \brief      Trigger a redraw for this renderer
     * \ingroup    RendererAPI
     */
    virtual void signalWindowSystemRedraw() = 0;

    /**
     * \brief      Force composition for entire scene
     * \ingroup    RendererAPI
     */
    virtual void forceCompositionWindowSystem() = 0;

    /**
      * \brief      Get the InputManager associated to the Scene
      * \ingroup    RendererAPI
      */
    virtual InputManager* getInputManager() const = 0;

    /**
      * \brief      Get the InputManager associated to the Scene
      * \ingroup    RendererAPI
      */
    virtual InputManager* getInputManager(char *pSeatName) const = 0;
    /**
      * \brief      Get the seat which have devices specified in the bitmask
      * \ingroup    RendererAPI
      */
    virtual bool getSupportedSeats(ilmInputDevice bitmask, t_ilm_string ** pSeatArray, unsigned int *pSizeSeatArray) const = 0;
    /**
     * \brief      Set the mode for the specified optimization (e.g. OFF,ON,AUTO)
     * \ingroup    RendererAPI
     * \param[in]  id id of optimization
     * \param[in] mode mode to set for the optimization
     * \return     TRUE: id and mode are valid and mode was set
     * \return     FALSE: id or mode was invalid and/or mode could not be set
     */
    virtual bool setOptimizationMode(OptimizationType id, OptimizationModeType mode) = 0;

    /**
     * \brief      Get the current mode for the specified optimization
     * \ingroup    RendererAPI
     * \param[in]  id id of optimization
     * \param[out] mode retrieved mode value
     * \return     TRUE: id is valid and mode was returned
     * \return     FALSE: id was invalid and/or mode was not returned
     */
    virtual bool getOptimizationMode(OptimizationType id, OptimizationModeType *mode) = 0;

    virtual bool isSurfaceDirty(unsigned int surfID) = 0;

    virtual bool isScreenFrozen(unsigned int screenID) = 0;

    /**
     * \brief      Set the gamma value of the provided screen ID
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \param[in]  value Gamma value to be set
     * \return     TRUE: Gamma value could be set successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID
     *                    or gamma setting is not supported
     */
    virtual bool setGamma(unsigned int screenID, double value) = 0;

    /**
     * \brief      Set the CSC values (brightness, contrast, hue, etc)to the screen ID
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \param[in]  pclrprop CSC values to be set
     * \return     TRUE: CSC values could be set successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID
     *                    or CSC settings may be failed
     */

    virtual bool setCSC(unsigned int screenID, ilmCSCProperties* pclrprop) = 0;

    virtual bool SetDisplayState(unsigned int screenId, ilmScreenState screenState) const = 0;

    virtual bool GetDisplayState(unsigned int screenId, ilmScreenState *screenState) const = 0;

    virtual bool SetDisplayAlpha(unsigned int screenId, unsigned int alphaValue) const = 0;

    virtual bool GetDisplayAlpha(unsigned int screenId, unsigned int *alphaValue) const = 0;

    virtual void destroyClientBufferShm(Surface *ilm_surface) = 0;

    /**
     * \brief      create vnc surface buffer for the provided display/screen ID
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \param[in]  bpp surface pixel format
     * \param[out]  pp_vncsurfptr vnc surface pointer
     * \param[out]  pp_buffaddr vnc surface logical buffer address
     * \return     TRUE: buffer could be created successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID
     *                    or bpp is not supported
     */
    virtual bool createVncbuffer(int screenID,int bpp,void **pp_vncsurfptr,void **pp_buffaddr) = 0;

    /**
     * \brief      destroy vnc surface
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \param[out] p_vncsurfptr vnc surface pointer
     * \return     TRUE: surface could be destroyed successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID
     *                    or surface ptr
     */
    virtual bool destroyVncbuffer(int screenID,void *p_vncsurfptr) = 0;

    /**
     * \brief      dump the provided screen data to vnc surf associated with screen ID
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \return     TRUE: VNC surface could be dumped successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID
     */
    virtual bool dumpFramebuffertovncsurf(int screenID) = 0;

    /**
     * \brief      process the touch events
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \param[in]  x touch co-ordinate x axis value
     * \param[in]  y touch co-ordinate y axis value
     * \param[in]  down - pressed or released value
     * \return     TRUE: events could be processed successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID or out of bounary values
     */
    virtual bool processVnctouchevent(int screenID,int x,int y,int down) = 0;

    /**
     * \brief      process the touch events
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \param[in]  x touch co-ordinate x axis value
     * \param[in]  y touch co-ordinate y axis value
     * \param[in]  buttonmask - pressed or released value
     * \return     TRUE: events could be processed successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID or out of bounary values
    */
    virtual bool processVncmouseevent(int screenID,int x,int y,int buttonmask) = 0;

    /**
     * \brief      process the touch events
     * \ingroup    RendererAPI
     * \param[in]  screenID screenID of display
     * \param[in]  keycode key button value
     * \param[in]  down - down or up value
     * \return     TRUE: events could be processed successfully
     * \return     FALSE: Error occurred during execution, e.g. invalid screenID or out of bounary values
    */
    virtual bool processVnckeyboardevent(int screenID,unsigned int keycode, int down) = 0;

    /**
     * \brief      Get the FPS of a screen handled by this renderer
     * \ingroup    RendererAPI
     * \param[in]  screenID id of the screen
     * \return     FPS of screen
     */
    virtual float getScreenFPS(unsigned int screenID) = 0;
};

#endif /* _IRENDERER_H_ */

